// WiseMindAI Web Clipper Background Service Worker
// 处理右键菜单和消息通信

const DEFAULT_HOST_PORT = '127.0.0.1:38221';

// 检查是否是支持的 URL 协议
function isSupportedUrl(url) {
  if (!url) return false;
  const supportedSchemes = ['http://', 'https://', 'file://'];
  return supportedSchemes.some(scheme => url.startsWith(scheme));
}

// 监听消息
chrome.runtime.onMessage.addListener((message, sender, _sendResponse) => {
  if (message.type === 'WISEMIND_OPEN_POPUP') {
    // 获取发送消息的标签页（来自 content script）
    const tabId = sender?.tab?.id;

    if (tabId) {
      // 获取标签页信息
      chrome.tabs.get(tabId, (tab) => {
        if (tab && !isSupportedUrl(tab.url)) {
          // 系统页面不打开 popup
          return;
        }

        // 将标签页 ID 作为参数传递，以便 popup.js 可以正确获取标签页信息
        const url = chrome.runtime.getURL(`popup.html?tabId=${tabId}`);

        // 获取标签页所在的窗口信息以计算居中位置
        chrome.windows.get(tab.windowId, (window) => {
          const width = 400;
          const height = 500;
          const left = window.left + Math.floor((window.width - width) / 2);
          const top = window.top + Math.floor((window.height - height) / 2);

          // 打开 popup 窗口（通过创建新窗口）
          chrome.windows.create({
            url: url,
            type: 'popup',
            width: width,
            height: height,
            left: left,
            top: top,
          });
        });
      });
    } else {
      // 如果没有 tabId，回退到原来的逻辑
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        const tab = tabs && tabs[0];
        if (tab && !isSupportedUrl(tab.url)) {
          // 系统页面不打开 popup
          return;
        }
        // 打开 popup 窗口（通过创建新窗口）
        chrome.windows.create({
          url: chrome.runtime.getURL('popup.html'),
          type: 'popup',
          width: 400,
          height: 500,
        });
      });
    }
  }
});
const API_PATH = '/api/create-file';

// 构建完整的 API 端点
function buildEndpoint(hostPort) {
  if (!hostPort) return null;
  let raw = hostPort.trim();
  if (!raw) return null;

  if (!raw.startsWith('http://') && !raw.startsWith('https://')) {
    raw = `http://${raw}`;
  }

  try {
    const url = new URL(raw);
    const port = url.port || (raw.startsWith('https://') ? '443' : '80');
    const hostPortStr = `${url.hostname}:${port}`;
    return `http://${hostPortStr}${API_PATH}`;
  } catch {
    return null;
  }
}

// 加载保存的 host:port
function loadHostPort() {
  return new Promise(resolve => {
    if (!chrome.storage || !chrome.storage.sync) {
      resolve(DEFAULT_HOST_PORT);
      return;
    }
    // 优先从新配置读取，兼容旧配置
    chrome.storage.sync.get(['wisemindConfig', 'wisemindHostPort'], result => {
      const config = result.wisemindConfig || {};
      resolve(config.apiEndpoint || result.wisemindHostPort || DEFAULT_HOST_PORT);
    });
  });
}

// 保存链接到 WiseMindAI
async function saveToWiseMind(data) {
  const hostPort = await loadHostPort();
  const endpoint = buildEndpoint(hostPort);

  if (!endpoint) {
    console.error('Invalid endpoint configuration');
    return { success: false, error: 'API 端点配置错误' };
  }

  const payload = {
    filePath: data.url,
    name: data.title || data.url,
    size: 0,
    type: 'link',
    fileType: 'link',
    icon: data.icon || '',
    content: data.content || '',
  };

  try {
    const res = await fetch(endpoint, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(payload),
    });

    if (!res.ok) {
      const text = await res.text().catch(() => '');
      throw new Error(text || `HTTP ${res.status}`);
    }

    return { success: true };
  } catch (err) {
    console.error('Save to WiseMindAI failed:', err);
    return { success: false, error: err?.message || String(err) };
  }
}

// 格式化笔记标题
function formatNoteTitle(template, title, date, time) {
  if (!template) return title;
  return template
    .replace(/{title}/g, title || '')
    .replace(/{date}/g, date || new Date().toLocaleDateString('zh-CN'))
    .replace(/{time}/g, time || new Date().toLocaleTimeString('zh-CN'));
}

// 格式化笔记后缀
function formatNoteSuffix(template, title, url) {
  if (!template) return '';
  return template
    .replace(/{title}/g, title || '')
    .replace(/{url}/g, url || '');
}

// 保存笔记到 WiseMindAI（使用 /api/create-note）
async function saveNoteToWiseMind(data) {
  const hostPort = await loadHostPort();
  const endpoint = buildEndpoint(hostPort)?.replace('/api/create-file', '/api/create-note');

  if (!endpoint) {
    console.error('Invalid endpoint configuration');
    return { success: false, error: 'API 端点配置错误' };
  }

  // 加载配置
  const config = await new Promise(resolve => {
    chrome.storage.sync.get(['wisemindConfig'], result => {
      resolve(result.wisemindConfig || {});
    });
  });

  // 格式化标题
  const now = new Date();
  const formattedTitle = formatNoteTitle(
    config.noteTitleFormat || '{title}',
    data.title || '笔记',
    now.toLocaleDateString('zh-CN'),
    now.toLocaleTimeString('zh-CN')
  );

  // 格式化内容（添加后缀）
  let finalContent = data.content || '';
  // 检查内容末尾是否已经包含来源信息，避免重复添加
  const hasSourceInfo = finalContent.includes('来源：') || finalContent.includes('原文链接：');

  if (config.showNoteSuffix !== false && config.noteSuffixContent && !hasSourceInfo) {
    const suffix = formatNoteSuffix(
      config.noteSuffixContent || '来源：《{title}》\n原文链接：{url}',
      data.title || '',
      data.url || ''
    );
    if (suffix) {
      finalContent = finalContent ? `${finalContent}\n\n${suffix}` : suffix;
    }
  }

  const payload = {
    title: formattedTitle,
    content: finalContent,
    md: finalContent,
    text: data.text || finalContent,
  };

  try {
    const res = await fetch(endpoint, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(payload),
    });

    if (!res.ok) {
      const text = await res.text().catch(() => '');
      throw new Error(text || `HTTP ${res.status}`);
    }

    return { success: true };
  } catch (err) {
    console.error('Save note to WiseMindAI failed:', err);
    return { success: false, error: err?.message || String(err) };
  }
}

// 创建右键菜单
chrome.runtime.onInstalled.addListener(() => {
  createContextMenus();
});

// 创建或更新右键菜单
function createContextMenus() {
  // 先清除所有菜单
  chrome.contextMenus.removeAll(() => {
    // 创建"保存 WiseMindAI 笔记"菜单（有选中文本时显示）
    // 使用 documentUrlPatterns 限制只在 http/https 页面显示
    chrome.contextMenus.create({
      id: 'saveNote',
      title: '保存 WiseMindAI 笔记',
      contexts: ['selection'],
      documentUrlPatterns: ['http://*/*', 'https://*/*', 'file://*/*'],
    });

    // 创建"保存为知识卡片"菜单（有选中文本时显示）
    chrome.contextMenus.create({
      id: 'saveCard',
      title: '保存为知识卡片',
      contexts: ['selection'],
      documentUrlPatterns: ['http://*/*', 'https://*/*', 'file://*/*'],
    });

    // 创建"页面保存在 WiseMindAI"菜单（在页面右键时显示，但排除选中文本的情况）
    // 注意：Chrome 的 contextMenus API 限制，我们需要使用 'page' 上下文
    // 但会在点击时检查是否有选中文本
    chrome.contextMenus.create({
      id: 'savePage',
      title: '页面保存在 WiseMindAI',
      contexts: ['page', 'frame', 'editable'],
      documentUrlPatterns: ['http://*/*', 'https://*/*', 'file://*/*'],
    });
  });
}

// 监听标签页更新，在系统页面禁用图标
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (changeInfo.status === 'complete' && tab.url) {
    if (!isSupportedUrl(tab.url)) {
      // 在系统页面设置图标为禁用状态（使用 badge 提示）
      chrome.action.setBadgeText({ text: '×', tabId: tabId });
      chrome.action.setBadgeBackgroundColor({ color: '#dc2626', tabId: tabId });
      chrome.action.setTitle({ title: '系统页面不支持保存', tabId: tabId });
    } else {
      // 在支持的页面清除图标状态
      chrome.action.setBadgeText({ text: '', tabId: tabId });
      chrome.action.setTitle({ title: '保存到 WiseMindAI', tabId: tabId });
    }
  }
});

// 监听标签页激活，更新图标状态
chrome.tabs.onActivated.addListener((activeInfo) => {
  chrome.tabs.get(activeInfo.tabId, (tab) => {
    if (tab && tab.url) {
      if (!isSupportedUrl(tab.url)) {
        chrome.action.setBadgeText({ text: '×', tabId: tab.id });
        chrome.action.setBadgeBackgroundColor({ color: '#dc2626', tabId: tab.id });
        chrome.action.setTitle({ title: '系统页面不支持保存', tabId: tab.id });
      } else {
        chrome.action.setBadgeText({ text: '', tabId: tab.id });
        chrome.action.setTitle({ title: '保存到 WiseMindAI', tabId: tab.id });
      }
    }
  });
});

// 处理右键菜单点击
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  // 检查是否是支持的 URL（通过 documentUrlPatterns 已经限制，这里做二次检查）
  if (!tab?.url || !isSupportedUrl(tab.url)) {
    return;
  }

  if (info.menuItemId === 'saveCard') {
    // 保存选中文本为知识卡片
    const fallbackText = info.selectionText || '';
    if (!fallbackText.trim()) {
      return;
    }

    // 获取选区纯文本（不使用 HTML）
    let selectionText = fallbackText;
    try {
      const response = await chrome.tabs.sendMessage(tab.id, { type: 'WISEMIND_GET_SELECTION' });
      if (response && response.ok && response.text && response.text.trim()) {
        selectionText = response.text;
      }
    } catch (e) {
      console.warn('Failed to get selection, fallback to plain text:', e);
    }

    // 打开知识卡片保存对话框（居中显示）
    const url = chrome.runtime.getURL(`cardDialog.html?text=${encodeURIComponent(selectionText)}`);

    // 获取标签页所在的窗口信息以计算居中位置
    chrome.windows.get(tab.windowId, (window) => {
      const width = 500;
      const height = 650;
      const left = window.left + Math.floor((window.width - width) / 2);
      const top = window.top + Math.floor((window.height - height) / 2);

      chrome.windows.create({
        url: url,
        type: 'popup',
        width: width,
        height: height,
        left: left,
        top: top,
      });
    });
  } else if (info.menuItemId === 'saveNote') {
    // 保存选中文本为笔记，优先使用 contentScript 提供的 HTML
    const fallbackText = info.selectionText || '';
    if (!fallbackText.trim()) {
      return;
    }

    // 获取选区 HTML + 纯文本
    let selectionText = fallbackText;
    try {
      const response = await chrome.tabs.sendMessage(tab.id, { type: 'WISEMIND_GET_SELECTION' });
      if (response && response.ok && response.text && response.text.trim()) {
        selectionText = response.text;
      }
    } catch (e) {
      // 某些页面无法注入，退回纯文本
      console.warn('Failed to get selection text, fallback to plain text:', e);
    }

    // 获取页面信息
    const pageUrl = tab.url || '';
    const pageTitle = tab.title || '网页';

    // 不在这里添加来源信息，让 saveNoteToWiseMind 根据配置统一处理
    const result = await saveNoteToWiseMind({
      title: pageTitle || '无标题',
      content: selectionText,
      text: selectionText,
      url: pageUrl,
    });

    if (result.success) {
      // 显示成功通知
      try {
        chrome.notifications.create({
          type: 'basic',
          iconUrl: 'icons/icon48.png',
          title: 'WiseMindAI',
          message: '✓ 笔记已成功保存',
        });
      } catch (e) {
        console.log('笔记已成功保存');
      }
    } else {
      // 显示失败通知
      const errorMsg = result.error || '未知错误';
      try {
        chrome.notifications.create({
          type: 'basic',
          iconUrl: 'icons/icon48.png',
          title: 'WiseMindAI',
          message: `✗ 保存失败: ${errorMsg}`,
        });
      } catch (e) {
        console.error('保存失败:', errorMsg);
      }
    }
  } else if (info.menuItemId === 'savePage') {
    // 检查是否有选中文本，如果有则不应该执行保存页面操作
    // 因为用户应该使用"保存笔记"菜单
    const selectedText = info.selectionText || '';
    if (selectedText.trim()) {
      // 有选中文本时，应该使用保存笔记功能
      const result = await saveNoteToWiseMind({
        title: `${tab.title || '无标题'}`,
        content: selectedText,
      });

      if (result.success) {
        // 显示成功通知
        try {
          chrome.notifications.create({
            type: 'basic',
            iconUrl: 'icons/icon48.png',
            title: 'WiseMindAI',
            message: '✓ 笔记已成功保存',
          });
        } catch (e) {
          console.log('笔记已成功保存');
        }
      } else {
        // 显示失败通知
        const errorMsg = result.error || '未知错误';
        try {
          chrome.notifications.create({
            type: 'basic',
            iconUrl: 'icons/icon48.png',
            title: 'WiseMindAI',
            message: `✗ 保存失败: ${errorMsg}`,
          });
        } catch (e) {
          console.error('保存失败:', errorMsg);
        }
      }
      return;
    }

    // 保存页面为链接
    try {
      // 尝试获取页面纯文本内容
      let pageContent = '';
      try {
        const response = await chrome.tabs.sendMessage(tab.id, { type: 'WISEMIND_GET_PAGE_TEXT' });
        if (response && response.ok && response.text) {
          pageContent = response.text;
        }
      } catch (e) {
        // 某些页面无法注入 content script，忽略错误
        console.warn('Failed to get page text content:', e);
      }

      const result = await saveToWiseMind({
        url: tab.url || '',
        title: tab.title || '',
        icon: tab.favIconUrl || '',
        content: pageContent,
      });

      if (result && result.success) {
        // 显示成功通知
        try {
          chrome.notifications.create({
            type: 'basic',
            iconUrl: 'icons/icon48.png',
            title: 'WiseMindAI',
            message: '✓ 页面已成功保存',
          });
        } catch (e) {
          console.log('页面已成功保存');
        }
      } else {
        // 显示失败通知
        const errorMsg = (result && result.error) || '未知错误';
        try {
          chrome.notifications.create({
            type: 'basic',
            iconUrl: 'icons/icon48.png',
            title: 'WiseMindAI',
            message: `✗ 保存失败: ${errorMsg}`,
          });
        } catch (e) {
          console.error('保存失败:', errorMsg);
        }
      }
    } catch (err) {
      // 捕获异常，确保总是显示提示
      const errorMsg = err?.message || String(err) || '未知错误';
      try {
        chrome.notifications.create({
          type: 'basic',
          iconUrl: 'icons/icon48.png',
          title: 'WiseMindAI',
          message: `✗ 保存失败: ${errorMsg}`,
        });
      } catch (e) {
        console.error('保存失败:', errorMsg);
      }
    }
  }
});

