// Content script: 获取选中文本的 HTML 和纯文本，以及浮窗按钮

function getSelectionHtmlAndText() {
  const selection = window.getSelection();
  if (!selection || selection.rangeCount === 0) {
    return { html: '', text: '' };
  }

  const range = selection.getRangeAt(0);
  const container = document.createElement('div');
  container.appendChild(range.cloneContents());

  const html = container.innerHTML || '';
  const text = selection.toString() || '';

  return { html, text };
}

// 获取整个页面的纯文本内容
function getPageTextContent() {
  try {
    // 克隆 body 元素以避免修改原页面
    const bodyClone = document.body.cloneNode(true);

    // 移除 script 和 style 标签
    const scripts = bodyClone.querySelectorAll('script, style, noscript');
    scripts.forEach(el => el.remove());

    // 获取纯文本内容
    const text = bodyClone.innerText || bodyClone.textContent || '';

    // 清理多余的空白字符
    return text
      .replace(/\s+/g, ' ')  // 多个空白字符替换为单个空格
      .replace(/\n\s*\n/g, '\n')  // 多个换行替换为单个换行
      .trim();
  } catch (e) {
    console.error('Failed to get page text content:', e);
    return '';
  }
}

// 浮窗按钮
let floatingButton = null;
let closeButton = null;
let closeDialog = null;

// 检查域名是否匹配白名单模式
function isDomainInWhitelist(domain, whitelist) {
  if (!whitelist || !whitelist.trim()) {
    return false;
  }

  const patterns = whitelist.split('\n').map(line => line.trim()).filter(line => line);
  const hostname = new URL(window.location.href).hostname;

  for (const pattern of patterns) {
    // 支持通配符匹配
    const regexPattern = pattern
      .replace(/\./g, '\\.')
      .replace(/\*/g, '.*');
    const regex = new RegExp(`^${regexPattern}$`, 'i');

    if (regex.test(hostname)) {
      return true;
    }
  }

  return false;
}

// 检查当前网站是否被禁用
async function isCurrentSiteDisabled() {
  return new Promise(resolve => {
    chrome.storage.sync.get(['wisemindDisabledSites'], result => {
      const disabledSites = result.wisemindDisabledSites || {};
      const hostname = new URL(window.location.href).hostname;
      resolve(disabledSites[hostname] === true);
    });
  });
}

// 检查本次访问是否已关闭
// 注意：这个状态只在当前页面会话中有效，页面刷新后会清除
async function isCurrentVisitClosed() {
  // 使用 sessionStorage 来存储，页面刷新后自动清除
  const sessionKey = 'wisemind_current_visit_closed';
  return sessionStorage.getItem(sessionKey) === 'true';
}

// 创建关闭按钮
function createCloseButton() {
  if (closeButton) {
    return;
  }

  closeButton = document.createElement('div');
  closeButton.id = 'wisemind-close-button';
  closeButton.innerHTML = '×';
  closeButton.style.cssText = `
    position: absolute;
    top: -8px;
    left: -8px;
    width: 20px;
    height: 20px;
    border-radius: 50%;
    background: #ffffff;
    color: #6b7280;
    font-size: 16px;
    font-weight: bold;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
    z-index: 1000000;
    transition: all 0.2s;
    line-height: 1;
    opacity: 0;
    pointer-events: none;
  `;

  closeButton.addEventListener('mouseenter', () => {
    closeButton.style.background = '#ef4444';
    closeButton.style.color = '#ffffff';
    closeButton.style.transform = 'scale(1.1)';
  });

  closeButton.addEventListener('mouseleave', () => {
    closeButton.style.background = '#ffffff';
    closeButton.style.color = '#6b7280';
    closeButton.style.transform = 'scale(1)';
  });

  closeButton.addEventListener('click', (e) => {
    e.stopPropagation();
    showCloseDialog();
  });

  if (floatingButton) {
    floatingButton.appendChild(closeButton);
  }
}

// 移除关闭按钮
function removeCloseButton() {
  if (closeButton && closeButton.parentNode) {
    closeButton.parentNode.removeChild(closeButton);
    closeButton = null;
  }
}

// 创建关闭对话框
function createCloseDialog() {
  if (closeDialog) {
    return;
  }

  closeDialog = document.createElement('div');
  closeDialog.id = 'wisemind-close-dialog';
  closeDialog.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    z-index: 10000000;
    display: none;
    align-items: center;
    justify-content: center;
  `;

  const dialogContent = document.createElement('div');
  dialogContent.style.cssText = `
    background: #ffffff;
    border-radius: 12px;
    padding: 24px;
    width: 400px;
    max-width: 90%;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
    position: relative;
  `;

  const closeX = document.createElement('div');
  closeX.innerHTML = '×';
  closeX.style.cssText = `
    position: absolute;
    top: 12px;
    right: 12px;
    width: 24px;
    height: 24px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    font-size: 20px;
    color: #6b7280;
    transition: all 0.2s;
    border-radius: 4px;
  `;

  closeX.addEventListener('mouseenter', () => {
    closeX.style.background = '#f3f4f6';
    closeX.style.color = '#111827';
  });

  closeX.addEventListener('mouseleave', () => {
    closeX.style.background = 'transparent';
    closeX.style.color = '#6b7280';
  });

  closeX.addEventListener('click', () => {
    hideCloseDialog();
  });

  const title = document.createElement('h3');
  title.textContent = '关闭悬浮球';
  title.style.cssText = `
    margin: 0 0 20px 0;
    font-size: 18px;
    font-weight: 700;
    color: #111827;
  `;

  const optionsContainer = document.createElement('div');
  optionsContainer.style.cssText = `
    display: flex;
    flex-direction: column;
    gap: 12px;
    margin-bottom: 24px;
  `;

  const options = [
    { value: 'current', label: '本次关闭直到下次访问' },
    { value: 'site', label: '当前网站禁用(可在设置页开启)' },
    { value: 'permanent', label: '永久禁用(可在设置页开启)' },
  ];

  let selectedOption = 'current';

  options.forEach(option => {
    const optionDiv = document.createElement('div');
    optionDiv.style.cssText = `
      display: flex;
      align-items: center;
      gap: 10px;
      padding: 12px;
      border-radius: 8px;
      cursor: pointer;
      transition: all 0.2s;
    `;

    const radio = document.createElement('input');
    radio.type = 'radio';
    radio.name = 'closeOption';
    radio.value = option.value;
    radio.checked = option.value === 'current';
    radio.style.cssText = `
      width: 18px;
      height: 18px;
      cursor: pointer;
      accent-color: #57be72;
    `;

    const label = document.createElement('label');
    label.textContent = option.label;
    label.style.cssText = `
      flex: 1;
      font-size: 14px;
      color: #374151;
      cursor: pointer;
      margin: 0;
    `;

    radio.addEventListener('change', () => {
      if (radio.checked) {
        selectedOption = option.value;
        optionsContainer.querySelectorAll('div').forEach(div => {
          div.style.background = '';
        });
        optionDiv.style.background = '#f0fdf4';
      }
    });

    // 初始化第一个选项的背景色
    if (option.value === 'current') {
      optionDiv.style.background = '#f0fdf4';
    }

    optionDiv.addEventListener('click', () => {
      radio.checked = true;
      radio.dispatchEvent(new Event('change'));
    });

    optionDiv.appendChild(radio);
    optionDiv.appendChild(label);
    optionsContainer.appendChild(optionDiv);
  });

  const buttonContainer = document.createElement('div');
  buttonContainer.style.cssText = `
    display: flex;
    gap: 10px;
    justify-content: flex-end;
  `;

  const cancelBtn = document.createElement('button');
  cancelBtn.textContent = '取消';
  cancelBtn.style.cssText = `
    padding: 10px 20px;
    font-size: 14px;
    font-weight: 600;
    border-radius: 8px;
    border: 2px solid #57be72;
    background: #ffffff;
    color: #57be72;
    cursor: pointer;
    transition: all 0.2s;
  `;

  cancelBtn.addEventListener('mouseenter', () => {
    cancelBtn.style.background = '#f0fdf4';
  });

  cancelBtn.addEventListener('mouseleave', () => {
    cancelBtn.style.background = '#ffffff';
  });

  cancelBtn.addEventListener('click', () => {
    hideCloseDialog();
  });

  const confirmBtn = document.createElement('button');
  confirmBtn.textContent = '确定';
  confirmBtn.style.cssText = `
    padding: 10px 20px;
    font-size: 14px;
    font-weight: 600;
    border-radius: 8px;
    border: none;
    background: #57be72;
    color: #ffffff;
    cursor: pointer;
    transition: all 0.2s;
  `;

  confirmBtn.addEventListener('mouseenter', () => {
    confirmBtn.style.background = '#4a9d5f';
  });

  confirmBtn.addEventListener('mouseleave', () => {
    confirmBtn.style.background = '#57be72';
  });

  confirmBtn.addEventListener('click', async () => {
    await handleCloseOption(selectedOption);
    hideCloseDialog();
  });

  buttonContainer.appendChild(cancelBtn);
  buttonContainer.appendChild(confirmBtn);

  dialogContent.appendChild(closeX);
  dialogContent.appendChild(title);
  dialogContent.appendChild(optionsContainer);
  dialogContent.appendChild(buttonContainer);

  closeDialog.appendChild(dialogContent);

  // 点击背景关闭
  closeDialog.addEventListener('click', (e) => {
    if (e.target === closeDialog) {
      hideCloseDialog();
    }
  });

  document.body.appendChild(closeDialog);
}

// 显示关闭对话框
function showCloseDialog() {
  if (!closeDialog) {
    createCloseDialog();
  }
  closeDialog.style.display = 'flex';
}

// 隐藏关闭对话框
function hideCloseDialog() {
  if (closeDialog) {
    closeDialog.style.display = 'none';
  }
}

// 处理关闭选项
async function handleCloseOption(option) {
  const url = window.location.href;
  const hostname = new URL(url).hostname;

  if (option === 'current') {
    // 本次关闭直到下次访问 - 使用 sessionStorage，页面刷新后自动恢复
    const sessionKey = 'wisemind_current_visit_closed';
    sessionStorage.setItem(sessionKey, 'true');
    removeFloatingButton();
  } else if (option === 'site') {
    // 当前网站禁用
    chrome.storage.sync.get(['wisemindDisabledSites'], result => {
      const disabledSites = result.wisemindDisabledSites || {};
      disabledSites[hostname] = true;
      chrome.storage.sync.set({ wisemindDisabledSites: disabledSites });
    });
    removeFloatingButton();
  } else if (option === 'permanent') {
    // 永久禁用
    chrome.storage.sync.get(['wisemindConfig'], result => {
      const config = result.wisemindConfig || {};
      config.enableFloatingButton = false;
      chrome.storage.sync.set({ wisemindConfig: config });
    });
    removeFloatingButton();
  }
}

// 移除关闭对话框
function removeCloseDialog() {
  if (closeDialog && closeDialog.parentNode) {
    closeDialog.parentNode.removeChild(closeDialog);
    closeDialog = null;
  }
}

function createFloatingButton() {
  if (floatingButton) {
    return;
  }

  // 加载保存的位置
  chrome.storage.local.get(['wisemindFloatingButtonPosition'], result => {
    const savedPosition = result.wisemindFloatingButtonPosition || { bottom: 20, right: 20, side: 'right' };

    // 确保默认位置是右下角
    if (!result.wisemindFloatingButtonPosition) {
      savedPosition.bottom = 20;
      savedPosition.right = 20;
      savedPosition.side = 'right';
    }

    floatingButton = document.createElement('div');
    floatingButton.id = 'wisemind-floating-button';

    // 根据保存的位置设置样式
    const positionStyle = savedPosition.side === 'left'
      ? `left: ${savedPosition.left || 0}px; right: auto;`
      : `right: ${savedPosition.right || 20}px; left: auto;`;

    floatingButton.style.cssText = `
      position: fixed;
      bottom: ${savedPosition.bottom || 20}px;
      ${positionStyle}
      width: 42px;
      height: 42px;
      border-radius: 50%;
      background: #57be72;
      box-shadow: 0 4px 12px rgba(87, 190, 114, 0.4);
      cursor: move;
      z-index: 999999;
      display: flex;
      align-items: center;
      justify-content: center;
      transition: all 0.3s;
      user-select: none;
      -webkit-user-select: none;
      -moz-user-select: none;
      -ms-user-select: none;
      -webkit-touch-callout: none;
      -webkit-tap-highlight-color: transparent;
    `;

    // 设置 draggable 为 false，防止浏览器默认拖拽行为
    floatingButton.setAttribute('draggable', 'false');

    const img = document.createElement('img');
    img.src = chrome.runtime.getURL('icons/icon48.png');
    img.style.cssText = `
      width: 30px;
      height: 30px;
      object-fit: contain;
      pointer-events: none;
    `;
    floatingButton.appendChild(img);

    // 创建关闭按钮
    createCloseButton();

    // 鼠标悬停时显示关闭按钮
    floatingButton.addEventListener('mouseenter', () => {
      floatingButton.style.transform = 'scale(1.1)';
      floatingButton.style.boxShadow = '0 6px 16px rgba(87, 190, 114, 0.5)';
      if (closeButton) {
        closeButton.style.opacity = '1';
        closeButton.style.pointerEvents = 'auto';
      }
    });

    floatingButton.addEventListener('mouseleave', () => {
      floatingButton.style.transform = 'scale(1)';
      floatingButton.style.boxShadow = '0 4px 12px rgba(87, 190, 114, 0.4)';
      if (closeButton) {
        closeButton.style.opacity = '0';
        closeButton.style.pointerEvents = 'none';
      }
    });

    // 拖拽功能
    let isDragging = false;
    let dragStartX = 0;
    let dragStartY = 0;
    let startLeft = 0;
    let startTop = 0;
    let hasMoved = false;
    let mouseMoveHandler = null;
    let mouseUpHandler = null;

    floatingButton.addEventListener('mousedown', (e) => {
      if (e.target === closeButton || closeButton?.contains(e.target)) {
        return; // 如果点击的是关闭按钮，不触发拖拽
      }

      isDragging = false;
      hasMoved = false;
      floatingButton.style.cursor = 'grabbing';
      floatingButton.style.transition = 'none';

      const rect = floatingButton.getBoundingClientRect();
      dragStartX = e.clientX;
      dragStartY = e.clientY;
      startLeft = rect.left;
      startTop = rect.top;

      // 阻止默认行为和事件冒泡，防止文本选择
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation();

      // 阻止文本选择
      if (window.getSelection) {
        window.getSelection().removeAllRanges();
      }

      // 只在 mousedown 时添加 mousemove 和 mouseup 监听器
      mouseMoveHandler = (moveEvent) => {
        if (!floatingButton) return;

        // 阻止默认行为和文本选择
        moveEvent.preventDefault();
        moveEvent.stopPropagation();

        const deltaX = Math.abs(moveEvent.clientX - dragStartX);
        const deltaY = Math.abs(moveEvent.clientY - dragStartY);

        // 如果移动距离超过阈值，开始拖拽
        if (deltaX > 3 || deltaY > 3) {
          if (!isDragging) {
            isDragging = true;
            hasMoved = true;
          }

          const moveDeltaX = moveEvent.clientX - dragStartX;
          const moveDeltaY = moveEvent.clientY - dragStartY;

          let newLeft = startLeft + moveDeltaX;
          let newTop = startTop + moveDeltaY;

          // 限制在可视区域内（使用实际大小 42px）
          const maxLeft = window.innerWidth - 42;
          const maxTop = window.innerHeight - 42;
          newLeft = Math.max(0, Math.min(newLeft, maxLeft));
          newTop = Math.max(0, Math.min(newTop, maxTop));

          floatingButton.style.left = newLeft + 'px';
          floatingButton.style.top = newTop + 'px';
          floatingButton.style.right = 'auto';
          floatingButton.style.bottom = 'auto';

          // 继续阻止文本选择
          if (window.getSelection) {
            window.getSelection().removeAllRanges();
          }
        }
      };

      mouseUpHandler = (upEvent) => {
        if (!floatingButton) return;

        // 阻止默认行为
        upEvent.preventDefault();
        upEvent.stopPropagation();

        // 移除事件监听器
        if (mouseMoveHandler) {
          document.removeEventListener('mousemove', mouseMoveHandler);
        }
        if (mouseUpHandler) {
          document.removeEventListener('mouseup', mouseUpHandler);
        }
        mouseMoveHandler = null;
        mouseUpHandler = null;

        if (isDragging) {
          floatingButton.style.cursor = 'move';
          floatingButton.style.transition = 'all 0.3s';

          // 自动吸附到左右两边
          const rect = floatingButton.getBoundingClientRect();
          const centerX = rect.left + rect.width / 2;
          const windowCenterX = window.innerWidth / 2;

          let finalLeft, finalRight, finalBottom, finalSide;

          if (centerX < windowCenterX) {
            // 吸附到左边
            finalLeft = 0;
            finalRight = 'auto';
            finalSide = 'left';
          } else {
            // 吸附到右边
            finalRight = 0;
            finalLeft = 'auto';
            finalSide = 'right';
          }

          // 保持垂直位置，但限制在可视区域内（使用实际大小 42px）
          const currentTop = rect.top;
          finalBottom = window.innerHeight - currentTop - 42;
          finalBottom = Math.max(0, Math.min(finalBottom, window.innerHeight - 42));

          floatingButton.style.left = finalLeft === 'auto' ? 'auto' : finalLeft + 'px';
          floatingButton.style.right = finalRight === 'auto' ? 'auto' : finalRight + 'px';
          floatingButton.style.top = 'auto';
          floatingButton.style.bottom = finalBottom + 'px';

          // 保存位置
          const position = {
            bottom: finalBottom,
            [finalSide]: finalSide === 'left' ? finalLeft : finalRight,
            side: finalSide
          };
          chrome.storage.local.set({ wisemindFloatingButtonPosition: position });
        } else if (!hasMoved) {
          // 如果没有移动，认为是点击
          chrome.runtime.sendMessage({
            type: 'WISEMIND_OPEN_POPUP',
          });
        }

        isDragging = false;
        hasMoved = false;
      };

      // 添加事件监听器
      document.addEventListener('mousemove', mouseMoveHandler);
      document.addEventListener('mouseup', mouseUpHandler);
    });

    // 窗口大小改变时调整位置
    const handleResize = () => {
      if (!floatingButton) return;
      const rect = floatingButton.getBoundingClientRect();

      // 确保悬浮球在可视区域内（使用实际大小 42px）
      if (rect.bottom > window.innerHeight) {
        const newBottom = Math.max(0, window.innerHeight - 42);
        floatingButton.style.bottom = newBottom + 'px';
        chrome.storage.local.get(['wisemindFloatingButtonPosition'], posResult => {
          const position = posResult.wisemindFloatingButtonPosition || savedPosition;
          position.bottom = newBottom;
          chrome.storage.local.set({ wisemindFloatingButtonPosition: position });
        });
      }
    };

    window.addEventListener('resize', handleResize);

    document.body.appendChild(floatingButton);
  });
}

function removeFloatingButton() {
  if (floatingButton && floatingButton.parentNode) {
    floatingButton.parentNode.removeChild(floatingButton);
    floatingButton = null;
  }
  removeCloseButton();
  removeCloseDialog();
}

// 检查是否启用浮窗按钮
async function checkFloatingButton() {
  // content_scripts 默认只会在支持的页面运行（通过 manifest.json 的 matches 配置）
  // 所以这里不需要额外检查
  chrome.storage.sync.get(['wisemindConfig'], async result => {
    const config = result.wisemindConfig || {};

    // 检查是否启用浮窗按钮
    if (!config.enableFloatingButton) {
      removeFloatingButton();
      return;
    }

    // 检查白名单
    if (isDomainInWhitelist(window.location.href, config.floatingButtonWhitelist)) {
      removeFloatingButton();
      return;
    }

    // 检查当前网站是否被禁用
    const isDisabled = await isCurrentSiteDisabled();
    if (isDisabled) {
      removeFloatingButton();
      return;
    }

    // 检查本次访问是否已关闭
    const isClosed = await isCurrentVisitClosed();
    if (isClosed) {
      removeFloatingButton();
      return;
    }

    // 所有检查通过，显示浮窗按钮
    createFloatingButton();
  });
}

// 监听配置变化
chrome.storage.onChanged.addListener((changes, areaName) => {
  if (areaName === 'sync') {
    if (changes.wisemindConfig) {
      checkFloatingButton();
    }
    if (changes.wisemindDisabledSites) {
      checkFloatingButton();
    }
  }
});

// 监听页面导航，清除本次关闭状态（使用 sessionStorage，页面刷新后自动清除，无需手动清除）
let lastUrl = window.location.href;
const observer = new MutationObserver(() => {
  const currentUrl = window.location.href;
  if (currentUrl !== lastUrl) {
    lastUrl = currentUrl;
    // URL 变化时，重新检查浮窗按钮
    setTimeout(() => {
      checkFloatingButton();
    }, 100);
  }
});

// 监听 URL 变化（适用于单页应用）
if (window.history && window.history.pushState) {
  const originalPushState = window.history.pushState;
  window.history.pushState = function (...args) {
    originalPushState.apply(window.history, args);
    setTimeout(() => {
      const currentUrl = window.location.href;
      if (currentUrl !== lastUrl) {
        lastUrl = currentUrl;
        setTimeout(() => {
          checkFloatingButton();
        }, 100);
      }
    }, 0);
  };
}

observer.observe(document.body, {
  childList: true,
  subtree: true,
});

chrome.runtime.onMessage.addListener((message, _sender, sendResponse) => {
  if (message && message.type === 'WISEMIND_GET_SELECTION') {
    try {
      const { html, text } = getSelectionHtmlAndText();
      sendResponse({ ok: true, html, text });
    } catch (e) {
      sendResponse({ ok: false, error: (e && e.message) || String(e) });
    }
    return true;
  }

  if (message && message.type === 'WISEMIND_GET_PAGE_TEXT') {
    try {
      const text = getPageTextContent();
      sendResponse({ ok: true, text });
    } catch (e) {
      sendResponse({ ok: false, error: (e && e.message) || String(e) });
    }
    return true;
  }

  if (message && message.type === 'WISEMIND_CONFIG_UPDATED') {
    checkFloatingButton();
  }
});

// 页面加载时清除"本次关闭"状态（刷新后恢复显示）
const sessionKey = 'wisemind_current_visit_closed';
sessionStorage.removeItem(sessionKey);

// 初始化浮窗按钮
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', checkFloatingButton);
} else {
  checkFloatingButton();
}


