// Options 页面脚本

const DEFAULT_HOST_PORT = '127.0.0.1:38221';
const API_PATH_CARD_DECKS = '/api/card-decks';

// 卡片背景色选项（参考知识卡片配置）
const CARD_COLORS = [
  { id: 'yellow', hex: '#fef08a', name: '黄色' },
  { id: 'blue', hex: '#bfdbfe', name: '蓝色' },
  { id: 'green', hex: '#bbf7d0', name: '绿色' },
  { id: 'pink', hex: '#fbcfe8', name: '粉色' },
  { id: 'purple', hex: '#e9d5ff', name: '紫色' },
  { id: 'gray', hex: '#e5e7eb', name: '灰色' },
  { id: 'orange', hex: '#fed7aa', name: '橙色' },
  { id: 'teal', hex: '#99f6e4', name: '青色' },
];

// 默认配置
const DEFAULT_CONFIG = {
  apiEndpoint: DEFAULT_HOST_PORT,
  enableFloatingButton: false,
  floatingButtonWhitelist: '',
  noteTitleFormat: '{title}',
  showNoteSuffix: true,
  noteSuffixContent: '来源：《{title}》\n原文链接：{url}',
  defaultDeck: '',
  defaultFolder: '',
  cardBackgroundColor: 'yellow',
};

// DOM 元素
const apiEndpoint = document.getElementById('apiEndpoint');
const enableFloatingButton = document.getElementById('enableFloatingButton');
const floatingButtonWhitelist = document.getElementById('floatingButtonWhitelist');
const noteTitleFormat = document.getElementById('noteTitleFormat');
const showNoteSuffix = document.getElementById('showNoteSuffix');
const noteSuffixContent = document.getElementById('noteSuffixContent');
const suffixContentGroup = document.getElementById('suffixContentGroup');
const defaultDeck = document.getElementById('defaultDeck');
const defaultFolder = document.getElementById('defaultFolder');
const cardColorPicker = document.getElementById('cardColorPicker');
const saveBtn = document.getElementById('saveBtn');
const resetBtn = document.getElementById('resetBtn');
const testApiBtn = document.getElementById('testApiBtn');
const statusEl = document.getElementById('status');

let cardDecks = [];
let currentConfig = { ...DEFAULT_CONFIG };

// 显示状态消息
function showStatus(message, type = 'success') {
  statusEl.textContent = message;
  statusEl.className = `status ${type} show`;
  setTimeout(() => {
    statusEl.classList.remove('show');
  }, 3000);
}

// 加载配置
function loadConfig() {
  return new Promise(resolve => {
    if (!chrome.storage || !chrome.storage.sync) {
      resolve(DEFAULT_CONFIG);
      return;
    }
    chrome.storage.sync.get(['wisemindConfig'], result => {
      currentConfig = { ...DEFAULT_CONFIG, ...(result.wisemindConfig || {}) };
      resolve(currentConfig);
    });
  });
}

// 保存配置
function saveConfig() {
  if (!chrome.storage || !chrome.storage.sync) return;
  
  currentConfig = {
    apiEndpoint: apiEndpoint.value.trim() || DEFAULT_HOST_PORT,
    enableFloatingButton: enableFloatingButton.checked,
    floatingButtonWhitelist: floatingButtonWhitelist.value.trim() || '',
    noteTitleFormat: noteTitleFormat.value.trim() || DEFAULT_CONFIG.noteTitleFormat,
    showNoteSuffix: showNoteSuffix.checked,
    noteSuffixContent: noteSuffixContent.value.trim() || DEFAULT_CONFIG.noteSuffixContent,
    defaultDeck: defaultDeck.value || '',
    defaultFolder: defaultFolder.value || '',
    cardBackgroundColor: currentConfig.cardBackgroundColor || 'yellow',
  };

  chrome.storage.sync.set({ wisemindConfig: currentConfig });
  
  // 同时保存 API 端点（兼容旧代码）
  chrome.storage.sync.set({ wisemindHostPort: currentConfig.apiEndpoint });
  
  // 通知所有标签页更新浮窗按钮状态
  chrome.tabs.query({}, tabs => {
    tabs.forEach(tab => {
      chrome.tabs.sendMessage(tab.id, { type: 'WISEMIND_CONFIG_UPDATED' }).catch(() => {
        // 忽略无法注入的页面
      });
    });
  });
  
  // 刷新已禁用网站列表
  loadDisabledSites();
  
  showStatus('✓ 设置已保存', 'success');
}

// 加载已禁用的网站列表
function loadDisabledSites() {
  const disabledSitesList = document.getElementById('disabledSitesList');
  if (!disabledSitesList) return;

  chrome.storage.sync.get(['wisemindDisabledSites'], result => {
    const disabledSites = result.wisemindDisabledSites || {};
    const sites = Object.keys(disabledSites).filter(site => disabledSites[site] === true);

    if (sites.length === 0) {
      disabledSitesList.innerHTML = '暂无已禁用的网站';
      disabledSitesList.style.color = '#6b7280';
      return;
    }

    disabledSitesList.innerHTML = '';
    disabledSitesList.style.color = '#111827';

    sites.forEach(site => {
      const siteItem = document.createElement('div');
      siteItem.style.cssText = `
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 8px 12px;
        margin-bottom: 6px;
        background: #ffffff;
        border: 1px solid #e5e7eb;
        border-radius: 6px;
        font-size: 13px;
      `;

      const siteName = document.createElement('span');
      siteName.textContent = site;
      siteName.style.cssText = 'color: #374151;';

      const enableBtn = document.createElement('button');
      enableBtn.textContent = '启用';
      enableBtn.style.cssText = `
        padding: 4px 12px;
        font-size: 12px;
        font-weight: 600;
        border-radius: 6px;
        border: 1px solid #57be72;
        background: #ffffff;
        color: #57be72;
        cursor: pointer;
        transition: all 0.2s;
      `;

      enableBtn.addEventListener('mouseenter', () => {
        enableBtn.style.background = '#f0fdf4';
      });

      enableBtn.addEventListener('mouseleave', () => {
        enableBtn.style.background = '#ffffff';
      });

      enableBtn.addEventListener('click', () => {
        chrome.storage.sync.get(['wisemindDisabledSites'], result => {
          const disabledSites = result.wisemindDisabledSites || {};
          delete disabledSites[site];
          chrome.storage.sync.set({ wisemindDisabledSites: disabledSites }, () => {
            loadDisabledSites();
            // 通知所有标签页更新
            chrome.tabs.query({}, tabs => {
              tabs.forEach(tab => {
                chrome.tabs.sendMessage(tab.id, { type: 'WISEMIND_CONFIG_UPDATED' }).catch(() => {});
              });
            });
          });
        });
      });

      siteItem.appendChild(siteName);
      siteItem.appendChild(enableBtn);
      disabledSitesList.appendChild(siteItem);
    });
  });
}

// 应用配置到 UI
function applyConfig(config) {
  apiEndpoint.value = config.apiEndpoint || DEFAULT_HOST_PORT;
  enableFloatingButton.checked = config.enableFloatingButton || false;
  floatingButtonWhitelist.value = config.floatingButtonWhitelist || '';
  noteTitleFormat.value = config.noteTitleFormat || DEFAULT_CONFIG.noteTitleFormat;
  showNoteSuffix.checked = config.showNoteSuffix !== false;
  noteSuffixContent.value = config.noteSuffixContent || DEFAULT_CONFIG.noteSuffixContent;
  suffixContentGroup.style.display = showNoteSuffix.checked ? 'block' : 'none';
  
  // 设置卡片背景色
  const selectedColor = config.cardBackgroundColor || 'yellow';
  updateColorPicker(selectedColor);
  
  // 加载已禁用的网站列表
  loadDisabledSites();
  
  // 保存当前配置用于后续更新
  currentConfig = { ...config };
}

// 加载卡片集列表
async function loadCardDecks() {
  try {
    const hostPort = currentConfig.apiEndpoint || DEFAULT_HOST_PORT;
    const endpoint = buildEndpoint(hostPort, API_PATH_CARD_DECKS);

    if (!endpoint) {
      return;
    }

    const res = await fetch(endpoint, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
      },
    });

    if (!res.ok) {
      return;
    }

    const result = await res.json();
    if (result.ok && result.data) {
      cardDecks = result.data;
      updateDeckSelect();
    }
  } catch (err) {
    console.error('Failed to load card decks:', err);
  }
}

// 更新卡片集下拉框
function updateDeckSelect() {
  if (!defaultDeck) return;

  defaultDeck.innerHTML = '<option value="">请选择卡片集</option>';
  cardDecks.forEach(deck => {
    const option = document.createElement('option');
    option.value = deck.id;
    option.textContent = deck.title;
    if (deck.id === Number(currentConfig.defaultDeck)) {
      option.selected = true;
    }
    defaultDeck.appendChild(option);
  });

  // 如果已选择卡片集，更新文件夹列表
  if (currentConfig.defaultDeck) {
    updateFolderSelect();
  }
}

// 更新卡片目录下拉框
function updateFolderSelect() {
  if (!defaultFolder) return;

  defaultFolder.innerHTML = '<option value="">请选择卡片目录</option>';

  const deckId = Number(defaultDeck.value);
  if (!deckId) {
    defaultFolder.innerHTML = '<option value="">请先选择卡片集</option>';
    return;
  }

  const deck = cardDecks.find(d => d.id === deckId);
  if (deck && deck.folders) {
    deck.folders.forEach(folder => {
      const option = document.createElement('option');
      option.value = folder.id;
      option.textContent = folder.title;
      if (folder.id === Number(currentConfig.defaultFolder)) {
        option.selected = true;
      }
      defaultFolder.appendChild(option);
    });
  }
}

// 构建 API 端点
function buildEndpoint(hostPort, path) {
  if (!hostPort) return null;
  let raw = hostPort.trim();
  if (!raw) return null;

  if (!raw.startsWith('http://') && !raw.startsWith('https://')) {
    raw = `http://${raw}`;
  }

  try {
    const url = new URL(raw);
    const port = url.port || (raw.startsWith('https://') ? '443' : '80');
    const hostPortStr = `${url.hostname}:${port}`;
    return `http://${hostPortStr}${path}`;
  } catch {
    return null;
  }
}

// 测试 API 连接
async function testApiConnection() {
  if (!apiEndpoint) {
    showStatus('✗ API 端点输入框未找到', 'error');
    return;
  }

  const hostPort = apiEndpoint.value.trim() || DEFAULT_HOST_PORT;
  if (!hostPort) {
    showStatus('✗ 请输入 API 端点', 'error');
    return;
  }

  // 使用 card-decks 接口进行测试（GET 请求，相对简单）
  const endpoint = buildEndpoint(hostPort, API_PATH_CARD_DECKS);
  if (!endpoint) {
    showStatus('✗ API 端点格式不正确', 'error');
    return;
  }

  // 更新按钮状态
  const originalText = testApiBtn.textContent;
  testApiBtn.disabled = true;
  testApiBtn.textContent = '测试中...';

  try {
    const res = await fetch(endpoint, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
      },
    });

    if (res.ok) {
      showStatus('✓ 连接成功！API 端点配置正确', 'success');
    } else {
      const text = await res.text().catch(() => '');
      showStatus(`✗ 连接失败：HTTP ${res.status} ${text || ''}`, 'error');
    }
  } catch (err) {
    const errorMsg = err?.message || String(err);
    const isConnectionError =
      errorMsg.includes('Failed to fetch') ||
      errorMsg.includes('NetworkError') ||
      errorMsg.includes('ERR_CONNECTION_REFUSED') ||
      errorMsg.includes('ERR_INTERNET_DISCONNECTED') ||
      errorMsg.includes('ERR_NAME_NOT_RESOLVED');

    if (isConnectionError) {
      showStatus('✗ 连接失败：无法连接到 WiseMindAI 服务。请确保 WiseMindAI 客户端已启动并启用了本地 API 服务', 'error');
    } else {
      showStatus(`✗ 连接失败：${errorMsg}`, 'error');
    }
  } finally {
    testApiBtn.disabled = false;
    testApiBtn.textContent = originalText;
  }
}

// 初始化颜色选择器
function initColorPicker() {
  if (!cardColorPicker) return;

  CARD_COLORS.forEach(color => {
    const option = document.createElement('div');
    option.className = 'color-option';
    option.style.backgroundColor = color.hex;
    option.title = color.name;
    option.dataset.colorId = color.id;
    option.addEventListener('click', () => {
      updateColorPicker(color.id);
      currentConfig.cardBackgroundColor = color.id;
    });
    cardColorPicker.appendChild(option);
  });
}

// 更新颜色选择器选中状态
function updateColorPicker(colorId) {
  if (!cardColorPicker) return;
  
  const options = cardColorPicker.querySelectorAll('.color-option');
  options.forEach(option => {
    if (option.dataset.colorId === colorId) {
      option.classList.add('selected');
    } else {
      option.classList.remove('selected');
    }
  });
}

// 重置配置
function resetConfig() {
  if (confirm('确定要重置所有设置为默认值吗？')) {
    currentConfig = { ...DEFAULT_CONFIG };
    applyConfig(currentConfig);
    saveConfig();
  }
}

// 菜单切换
const menuItems = document.querySelectorAll('.menu-item');
const sectionContents = document.querySelectorAll('.section-content');
const contentTitle = document.getElementById('contentTitle');

const sectionTitles = {
  api: 'API 配置',
  document: '文档配置',
  note: '笔记配置',
  card: '知识卡片配置',
};

function switchSection(sectionId) {
  // 更新菜单项状态
  menuItems.forEach(item => {
    if (item.dataset.section === sectionId) {
      item.classList.add('active');
    } else {
      item.classList.remove('active');
    }
  });

  // 更新内容区
  sectionContents.forEach(section => {
    if (section.id === `${sectionId}Section`) {
      section.classList.add('active');
    } else {
      section.classList.remove('active');
    }
  });

  // 更新标题
  if (contentTitle) {
    contentTitle.textContent = sectionTitles[sectionId] || '设置';
  }

  // 如果切换到文档配置，刷新已禁用网站列表
  if (sectionId === 'document') {
    loadDisabledSites();
  }
}

// 初始化
async function init() {
  const config = await loadConfig();
  applyConfig(config);
  initColorPicker();
  await loadCardDecks();

  // 菜单项点击事件
  menuItems.forEach(item => {
    item.addEventListener('click', () => {
      switchSection(item.dataset.section);
    });
  });
}

// 事件监听
saveBtn?.addEventListener('click', () => {
  saveConfig();
});

resetBtn?.addEventListener('click', () => {
  resetConfig();
});

showNoteSuffix?.addEventListener('change', () => {
  if (suffixContentGroup) {
    suffixContentGroup.style.display = showNoteSuffix.checked ? 'block' : 'none';
  }
});

defaultDeck?.addEventListener('change', () => {
  currentConfig.defaultDeck = defaultDeck.value;
  updateFolderSelect();
});

defaultFolder?.addEventListener('change', () => {
  currentConfig.defaultFolder = defaultFolder.value;
});

apiEndpoint?.addEventListener('change', async () => {
  // API 端点改变时，重新加载卡片集
  currentConfig.apiEndpoint = apiEndpoint.value.trim() || DEFAULT_HOST_PORT;
  await loadCardDecks();
});

testApiBtn?.addEventListener('click', () => {
  testApiConnection();
});

// 初始化
init();

